<?php
namespace App\Services\CsvStreamServices;

use App\Exceptions\EmptyCsvDownloadException;
use App\Exceptions\EncodingCsvDownloadException;
use Illuminate\Support\LazyCollection;

/**
 * CSVダウンロードサービス抽象クラス（ストリーム版）
 * @author k-maeda@winas.jp
 */
abstract class AbstractCsvStreamService
{
    /**
     * CSVデータ(2行目以降)
     */
    private ?LazyCollection $csvData = null;

    /**
     * ファイル名取得
     * @param void
     * @return String
     */
    abstract protected function getFileName(): string;

    /**
     * 項目名データ取得
     * @param void
     * @return Array
     */
    abstract protected function getCsvHeader(): Array;

    /**
     * 項目名データ取得
     * @param void
     * @return Array
     */
    abstract protected function format($entity): Array;

    /**
     * CSVデータ取得
     * @param void
     * @return Array
     */
    public function setCsvData(LazyCollection $csvData)
    {
        $this->csvData = $csvData;
    }

    /**
     * Description
     * @return type
     */
    public function streamDownload()
    {
        $csv_header = $this->getCsvHeader();
        $file_name = $this->getFileName();
        $csv_data   = $this->csvData;

        if (count($csv_header) > 0) {
            $csv_header = $this->convertEncoding($csv_header); // SJIS-win
        }

        $callback = function () use ($csv_data, $csv_header) {
            $fp = fopen('php://output', 'w');
            fputcsv($fp, $csv_header);
            foreach ($csv_data as $record) {
                $row = $this->format($record);
                $row = $this->convertEncoding($row);        // SJIS-win
                $row = $this->convertCommaDoubleByte($row); // 半角カンマを全角へ変換
                $row = $this->deleteCRLF($row);             // 改行を削除
                fputcsv($fp, $row);
            }
            fclose($fp);
        };

        return response()->streamDownload($callback, $file_name . '.csv', $this->responseHeaders($file_name));
    }

    /**
     * ヘッダー取得
     * @param  String $fileName
     * @return Array
     */
    private function responseHeaders(String $fileName)
    {
        return [
                'Cache-Control'             => 'must-revalidate, post-check=0, pre-check=0'
            ,   'Content-type'              => 'text/csv'
            ,   'Content-Disposition'       => 'attachment; filename=' . $fileName . '.csv'
            ,   'Content-Transfer-Encoding' => 'binary'
            ,   'Expires'                   => '0'
            ,   'Pragma'                    => 'public'
        ];
    }

    /**
     * Shift-jis に変換
     * @param  Array  $record
     * @return Array  $record
     */
    private function convertEncoding(Array $record)
    {
        $code = mb_convert_variables('SJIS-win', 'UTF-8', $record);
        if ($code === false) {
            throw new EncodingCsvDownloadException('CSVのエンコード変換に失敗しました。');
        }
        return $record;
    }

    /**
     * 半角カンマ「,」がある場合、全角カンマ「，」へ変換
     * @param  Array  $record
     * @return Array  $result
     */
    private function convertCommaDoubleByte(Array $record)
    {
        $result = [];
        foreach ($record as $column) {
            $result[] = str_replace(',', '，', $column);
        }
        return $result;
    }

    /**
     * 改行がある場合、削除
     * @param  Array  $record
     * @return Array  $result
     */
    private function deleteCRLF(Array $record)
    {
        $result = [];
        foreach ($record as $column) {
             // すべての改行コードに対応していることを明確にするため、あえて全部記載している。
             // TODO なぜか効かないがCSVファイル上では\nという文字として扱われているため、今のところ問題なし。いずれ修正したい
            $result[] = str_replace(["\r\n", "\r", "\n"], '', $column);
        }
        return $result;
    }
}
